<?php
// authentication/classic/reset-password/change-password.php
session_start();

// --- Load DB config from demo1 root ---
$rootConfig = __DIR__ . '/../../../config.php';
if (!file_exists($rootConfig)) {
    http_response_code(500);
    echo '<div id="page-content" class="p-6 text-sm text-destructive-foreground">config.php not found.</div>';
    exit;
}
require_once $rootConfig;

$pdo = $pdo ?? null;
try {
    if (!$pdo instanceof PDO) {
        if (isset($DB_DSN, $DB_USER, $DB_PASS)) {
            $pdo = new PDO($DB_DSN, $DB_USER, $DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } elseif (isset($DB_HOST, $DB_NAME, $DB_USER, $DB_PASS)) {
            $dsn = "mysql:host={$DB_HOST};dbname={$DB_NAME};charset=utf8mb4";
            $pdo = new PDO($dsn, $DB_USER, $DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } elseif (isset($db) && is_array($db)) {
            $host = $db['host'] ?? '127.0.0.1';
            $name = $db['name'] ?? '';
            $user = $db['user'] ?? '';
            $pass = $db['pass'] ?? '';
            $dsn  = "mysql:host={$host};dbname={$name};charset=utf8mb4";
            $pdo  = new PDO($dsn, $user, $pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } else {
            throw new RuntimeException('No PDO or DB credentials exported by config.php');
        }
    }
} catch (Throwable $e) {
    http_response_code(500);
    echo '<div id="page-content" class="p-6 text-sm text-destructive-foreground">Database connection failed: '
        . htmlspecialchars($e->getMessage()) . '</div>';
    exit;
}

if (!function_exists('h')) {
    function h(?string $s): string { return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
if (!function_exists('client_ip')) {
    function client_ip(): string {
        foreach (['HTTP_CLIENT_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR'] as $k) {
            if (!empty($_SERVER[$k])) return explode(',', (string)$_SERVER[$k])[0];
        }
        return '';
    }
}

// --- Require auth ---
$currentUserId = $_SESSION['user_id'] ?? null;
if (!$currentUserId) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head><meta charset="utf-8"><title>Change Password</title></head>
    <body>
    <div id="page-content" class="p-6">
        <div class="max-w-md mx-auto">
            <div class="text-sm p-4 rounded-lg bg-accent/40 text-secondary-foreground">
                You must be logged in to change your password.
                <a href="/novademo/html/demo1/index.php#login" class="text-primary underline ml-1">Go to Login</a>
            </div>
        </div>
    </div>
    </body>
    </html>
    <?php
    exit;
}

// --- CSRF ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

// --- Messages ---
$messages = ['errors' => [], 'success' => []];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $token = $_POST['csrf_token'] ?? '';
    $new   = trim((string)($_POST['user_new_password'] ?? ''));
    $conf  = trim((string)($_POST['user_confirm_password'] ?? ''));

    if (!hash_equals($csrf, $token)) {
        $messages['errors'][] = 'Invalid security token. Please refresh and try again.';
    } else {
        if ($new === '' || $conf === '') {
            $messages['errors'][] = 'Both password fields are required.';
        } elseif ($new !== $conf) {
            $messages['errors'][] = 'Passwords do not match.';
        } elseif (strlen($new) < 8) {
            $messages['errors'][] = 'Password must be at least 8 characters.';
        }

        if (!$messages['errors']) {
            try {
                $pdo->beginTransaction();

                $hash = password_hash($new, PASSWORD_BCRYPT);

                $stmt = $pdo->prepare("
                    UPDATE users
                       SET password_hash = :hash,
                           updated_ip     = :ip,
                           updated_at     = NOW()
                     WHERE id = :id
                    LIMIT 1
                ");
                $stmt->execute([
                    ':hash' => $hash,
                    ':ip'   => client_ip(),
                    ':id'   => $currentUserId
                ]);

                $stmt2 = $pdo->prepare("
                    UPDATE password_resets
                       SET used_at = NOW()
                     WHERE user_id = :id
                       AND used_at IS NULL
                ");
                $stmt2->execute([':id' => $currentUserId]);
                $auditToken = bin2hex(random_bytes(16));
                $stmt3 = $pdo->prepare("
                    INSERT INTO password_resets (user_id, token, expires_at, used_at, created_at)
                    VALUES (:uid, :token, NULL, NOW(), NOW())
                ");
                $stmt3->execute([
                    ':uid'   => $currentUserId,
                    ':token' => $auditToken,
                ]);

                $pdo->commit();

                $messages['success'][] = 'Your password has been updated.';
                // rotate CSRF
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
                $csrf = $_SESSION['csrf_token'];
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $messages['errors'][] = 'Could not update password: ' . $e->getMessage();
            }
        }
    }
}
?>
<!DOCTYPE html>
<html class="h-full" data-kt-theme="true" data-kt-theme-mode="light" dir="ltr" lang="en">
<head>
  <meta charset="utf-8"/>
  <title>Change Password</title>
  <base href="../../../../../">
  <link href="assets/vendors/keenicons/styles.bundle.css" rel="stylesheet"/>
  <link href="assets/css/styles.css" rel="stylesheet"/>
</head>
<body class="antialiased flex h-full text-base text-foreground bg-background">
<div id="page-content" class="p-6 min-h-[70vh] flex items-center justify-center">
  <div class="kt-card w-full max-w-[370px]">
    <form
      id="reset_password_change_password_form"
      method="post"
      action="/novademo/html/demo1/authentication/classic/reset-password/change-password.php"
      class="kt-card-content flex flex-col gap-5 p-10"
      onsubmit="event.preventDefault();(async function(f){
          try{
              var root=document.querySelector('#page-content');
              if(root){ root.classList.add('opacity-70','pointer-events-none'); }
              var fd=new FormData(f);
              var res=await fetch(f.action,{method:'POST',body:fd,credentials:'same-origin',headers:{'X-Requested-With':'fetch'},cache:'no-store'});
              var txt=await res.text();
              var doc=new DOMParser().parseFromString(txt,'text/html');
              var fresh=doc.querySelector('#page-content');
              if(fresh&&root){ root.replaceWith(fresh); }
              // Keep the page inside index.php with #change-password
              var INDEX_URL='/novademo/html/demo1/index.php';
              if (history && history.pushState) {
                  history.pushState({page:'change-password',params:{}}, '', INDEX_URL + '#change-password');
              } else {
                  window.location.href = INDEX_URL + '#change-password';
              }
          }catch(e){
              var root=document.querySelector('#page-content');
              if(root){
                  var box=document.createElement('div');
                  box.className='mb-4 p-4 rounded-lg bg-destructive/10 text-destructive-foreground text-sm';
                  box.textContent='Failed to submit: ' + (e && e.message ? e.message : e);
                  root.prepend(box);
              }
          }finally{
              var root=document.querySelector('#page-content');
              if(root){ root.classList.remove('opacity-70','pointer-events-none'); }
          }
      })(this);"
    >
      <div class="text-center">
        <h3 class="text-lg font-medium text-mono">Reset Password</h3>
        <span class="text-sm text-secondary-foreground">Enter your new password</span>
      </div>

      <?php if ($messages['errors']): ?>
          <div class="p-4 rounded-lg bg-destructive/10 text-destructive-foreground text-sm">
              <ul class="list-disc ps-5">
                  <?php foreach ($messages['errors'] as $err): ?>
                      <li><?= h($err) ?></li>
                  <?php endforeach; ?>
              </ul>
          </div>
      <?php endif; ?>

      <?php if ($messages['success']): ?>
          <div class="p-4 rounded-lg bg-green-500/10 text-green-600 text-sm">
              <?php foreach ($messages['success'] as $ok): ?>
                  <div><?= h($ok) ?></div>
              <?php endforeach; ?>
          </div>
      <?php endif; ?>

      <input type="hidden" name="csrf_token" value="<?= h($csrf) ?>"/>

      <div class="flex flex-col gap-1">
        <label class="kt-form-label text-mono">New Password</label>
        <label class="kt-input" data-kt-toggle-password="true">
          <input name="user_new_password" placeholder="Enter a new password" type="password"
                 required minlength="8" pattern=".{8,}" value=""/>
          <div class="kt-btn kt-btn-sm kt-btn-ghost kt-btn-icon bg-transparent! -me-1.5" data-kt-toggle-password-trigger="true">
            <span class="kt-toggle-password-active:hidden"><i class="ki-filled ki-eye text-muted-foreground"></i></span>
            <span class="hidden kt-toggle-password-active:block"><i class="ki-filled ki-eye-slash text-muted-foreground"></i></span>
          </div>
        </label>
        <span class="text-xs text-muted-foreground">At least 8 characters.</span>
      </div>

      <div class="flex flex-col gap-1">
        <label class="kt-form-label font-normal text-mono">Confirm New Password</label>
        <label class="kt-input" data-kt-toggle-password="true">
          <input name="user_confirm_password" placeholder="Re-enter a new Password" type="password"
                 required minlength="8" pattern=".{8,}" value=""/>
          <div class="kt-btn kt-btn-sm kt-btn-ghost kt-btn-icon bg-transparent! -me-1.5" data-kt-toggle-password-trigger="true">
            <span class="kt-toggle-password-active:hidden"><i class="ki-filled ki-eye text-muted-foreground"></i></span>
            <span class="hidden kt-toggle-password-active:block"><i class="ki-filled ki-eye-slash text-muted-foreground"></i></span>
          </div>
        </label>
      </div>

      <button class="kt-btn kt-btn-primary flex justify-center">Submit</button>
    </form>
  </div>
</div>
</body>
</html>
