﻿<?php
// userdashboard/profile.php
session_start();

// --- Load DB config from demo1 root ---
$rootConfig = __DIR__ . '/../config.php';
if (!file_exists($rootConfig)) {
    http_response_code(500);
    echo '<div id="page-content" class="p-6 text-sm text-destructive-foreground">config.php not found in demo1 root.</div>';
    exit;
}
require_once $rootConfig;

$pdo = $pdo ?? null;
try {
    if (!$pdo instanceof PDO) {
        if (isset($DB_DSN, $DB_USER, $DB_PASS)) {
            $pdo = new PDO($DB_DSN, $DB_USER, $DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } elseif (isset($DB_HOST, $DB_NAME, $DB_USER, $DB_PASS)) {
            $dsn = "mysql:host={$DB_HOST};dbname={$DB_NAME};charset=utf8mb4";
            $pdo = new PDO($dsn, $DB_USER, $DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } elseif (isset($db) && is_array($db)) {
            $host = $db['host'] ?? '127.0.0.1';
            $name = $db['name'] ?? '';
            $user = $db['user'] ?? '';
            $pass = $db['pass'] ?? '';
            $dsn  = "mysql:host={$host};dbname={$name};charset=utf8mb4";
            $pdo  = new PDO($dsn, $user, $pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } else {
            throw new RuntimeException('No PDO or DB credentials exported by config.php');
        }
    }
} catch (Throwable $e) {
    http_response_code(500);
    echo '<div id="page-content" class="p-6 text-sm text-destructive-foreground">Database connection failed: '
        . htmlspecialchars($e->getMessage()) . '</div>';
    exit;
}

// --- Helpers ---
if (!function_exists('h')) {
    function h(?string $s): string { return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
if (!function_exists('only_digits')) {
    function only_digits(string $s): string { return preg_replace('/\D+/', '', $s); }
}
if (!function_exists('client_ip')) {
    function client_ip(): string {
        foreach (['HTTP_CLIENT_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR'] as $k) {
            if (!empty($_SERVER[$k])) return explode(',', (string)$_SERVER[$k])[0];
        }
        return '';
    }
}

$currentUserId = $_SESSION['user_id'] ?? null;
$currentUserEmail = $_SESSION['user_email'] ?? null;

if (!$currentUserId && $currentUserEmail) {
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = :email LIMIT 1");
    $stmt->execute([':email' => $currentUserEmail]);
    $row = $stmt->fetch();
    if ($row) $currentUserId = (int)$row['id'];
}

$messages = ['errors' => [], 'success' => []];

if (!$currentUserId) {
    ?>
    <div id="page-content" class="w-full flex justify-center py-12">
      <div class="kt-card max-w-[600px] w-full">
        <div class="kt-card-content flex flex-col gap-5 p-10">
          <h2 class="text-xl font-semibold text-center">Profile</h2>
          <div class="text-sm p-4 rounded-lg bg-accent/40 text-secondary-foreground text-center">
            You must be logged in to view your profile.
            <a href="#login" data-page="login" class="text-primary underline ml-1">Go to Login</a>
          </div>
        </div>
      </div>
    </div>
    <?php
    exit;
}

// --- CSRF token ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

// --- Handle POST (update profile) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $token = $_POST['csrf_token'] ?? '';
    if (!hash_equals($csrf, $token)) {
        $messages['errors'][] = 'Invalid security token. Please refresh and try again.';
    } else {
        $first_name       = trim((string)($_POST['first_name'] ?? ''));
        $last_name        = trim((string)($_POST['last_name'] ?? ''));
        $country          = trim((string)($_POST['country'] ?? ''));
        $city             = trim((string)($_POST['city'] ?? ''));
        $address_line     = trim((string)($_POST['address_line'] ?? ''));
        $postal_code      = trim((string)($_POST['postal_code'] ?? ''));
        $phone            = trim((string)($_POST['phone'] ?? ''));
        $preferred_locale = trim((string)($_POST['preferred_locale'] ?? ''));

        if ($first_name === '') $messages['errors'][] = 'First name is required.';
        if ($last_name === '')  $messages['errors'][] = 'Last name is required.';
        if ($phone !== '' && !preg_match('/^[\d\s\-\+\(\)]+$/', $phone)) {
            $messages['errors'][] = 'Phone contains invalid characters.';
        }
        if ($preferred_locale !== '' && !in_array($preferred_locale, ['en','de','es'], true)) {
            $messages['errors'][] = 'Invalid preferred locale.';
        }

        if (!$messages['errors']) {
            $phone_digits = only_digits($phone);
            $ip = client_ip();

            try {
                $pdo->beginTransaction();

                $sets = [
                    "phone = :phone",
                    "phone_digits = :phone_digits",
                    "updated_ip = :updated_ip",
                    "updated_at = NOW()"
                ];
                $paramsU = [
                    ':phone'        => $phone,
                    ':phone_digits' => $phone_digits,
                    ':updated_ip'   => $ip,
                    ':id'           => $currentUserId
                ];
                if ($preferred_locale !== '') {
                    $sets[] = "preferred_locale = :pref";
                    $paramsU[':pref'] = $preferred_locale;
                }
                $sqlU = "UPDATE users SET " . implode(", ", $sets) . " WHERE id = :id";
                $stmtU = $pdo->prepare($sqlU);
                $stmtU->execute($paramsU);

                $exists = $pdo->prepare("SELECT 1 FROM user_profiles WHERE user_id = :id LIMIT 1");
                $exists->execute([':id' => $currentUserId]);

                if ($exists->fetchColumn()) {
                    $sqlP = "UPDATE user_profiles
                             SET first_name = :first_name,
                                 last_name  = :last_name,
                                 country    = :country,
                                 city       = :city,
                                 address_line = :address_line,
                                 postal_code  = :postal_code,
                                 updated_at   = NOW()
                             WHERE user_id = :id";
                } else {
                    $sqlP = "INSERT INTO user_profiles
                                (user_id, first_name, last_name, country, city, address_line, postal_code, created_at, updated_at)
                             VALUES
                                (:id, :first_name, :last_name, :country, :city, :address_line, :postal_code, NOW(), NOW())";
                }
                $stmtP = $pdo->prepare($sqlP);
                $stmtP->execute([
                    ':id'           => $currentUserId,
                    ':first_name'   => $first_name,
                    ':last_name'    => $last_name,
                    ':country'      => $country,
                    ':city'         => $city,
                    ':address_line' => $address_line,
                    ':postal_code'  => $postal_code,
                ]);

                $pdo->commit();
                $messages['success'][] = 'Profile updated successfully.';
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
                $csrf = $_SESSION['csrf_token'];
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $messages['errors'][] = 'Update failed: ' . $e->getMessage();
            }
        }
    }
}

// --- Load current data ---
$stmt = $pdo->prepare("
    SELECT u.id, u.email, u.phone, u.phone_digits, u.is_active, u.registration_channel,
           u.preferred_locale, u.affiliate_id, u.affiliate_code_value, u.user_source_type, u.user_source_label,
           u.current_status_id, u.current_status_set_at, u.current_assignment_id, u.current_desk_id,
           u.current_manager_user_id, u.current_agent_user_id, u.last_login_at, u.last_login_ip,
           u.created_ip, u.updated_ip, u.created_at, u.updated_at,
           up.first_name, up.last_name, up.country, up.city, up.address_line, up.postal_code, up.extra
    FROM users u
    LEFT JOIN user_profiles up ON up.user_id = u.id
    WHERE u.id = :id
    LIMIT 1
");
$stmt->execute([':id' => $currentUserId]);
$data = $stmt->fetch();

if (!$data) {
    ?>
    <div id="page-content" class="w-full flex justify-center py-12">
      <div class="kt-card max-w-[600px] w-full">
        <div class="kt-card-content flex flex-col gap-5 p-10">
          <h2 class="text-xl font-semibold text-center">Profile</h2>
          <div class="text-sm p-4 rounded-lg bg-destructive/10 text-destructive-foreground text-center">
            Profile not found.
          </div>
        </div>
      </div>
    </div>
    <?php
    exit;
}

$first_name       = $data['first_name'] ?? '';
$last_name        = $data['last_name'] ?? '';
$country          = $data['country'] ?? '';
$city             = $data['city'] ?? '';
$address_line     = $data['address_line'] ?? '';
$postal_code      = $data['postal_code'] ?? '';
$phone            = $data['phone'] ?? '';
$preferred_locale = $data['preferred_locale'] ?? 'en';
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8"/>
  <title>Profile - Nova Chain Investments</title>
</head>
<body class="antialiased flex h-full text-base text-foreground bg-background">
<main id="page-content" class="w-full flex justify-center py-12">
  <div class="kt-card max-w-[600px] w-full">
    <div class="kt-card-content flex flex-col gap-5 p-10">
      <div class="text-center">
        <h1 class="text-lg font-medium text-mono">Profile</h1>
        <span class="text-sm text-secondary-foreground">Update your information</span>
      </div>

      <div class="flex justify-center">
        <a id="changePasswordLink"
           href="/novademo/html/demo1/index.php#change-password"
           data-target="/novademo/html/demo1/authentication/classis/reset-password/change-password.php"
           class="kt-btn kt-btn-primary">
          Change Password
        </a>
      </div>

      <?php if ($messages['errors']): ?>
        <div class="rounded-lg border border-red-300 bg-red-50 p-4 text-red-800">
          <ul class="list-disc ml-5">
            <?php foreach ($messages['errors'] as $err): ?>
              <li><?= h($err) ?></li>
            <?php endforeach; ?>
          </ul>
        </div>
      <?php endif; ?>

      <?php if ($messages['success']): ?>
        <div class="rounded-lg border border-emerald-300 bg-emerald-50 p-4 text-emerald-800">
          <?php foreach ($messages['success'] as $ok): ?>
            <div><?= h($ok) ?></div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>

      <form
        id="profileForm"
        method="post"
        action="/novademo/html/demo1/userdashboard/profile.php"
        onsubmit="event.preventDefault();(async function(f){
            var INDEX_URL='/novademo/html/demo1/index.php';
            try{
                var root=document.querySelector('#page-content');
                if(root){ root.classList.add('opacity-70','pointer-events-none'); }
                var fd=new FormData(f);
                var res=await fetch(f.action,{method:'POST',body:fd,credentials:'same-origin',headers:{'X-Requested-With':'fetch'},cache:'no-store'});
                var txt=await res.text();
                var doc=new DOMParser().parseFromString(txt,'text/html');
                var fresh=doc.querySelector('#page-content');
                if(fresh&&root){ root.replaceWith(fresh); }
                if (history && history.pushState) {
                    history.pushState({page:'profile',params:{}}, '', INDEX_URL + '#profile');
                } else {
                    window.location.href = INDEX_URL + '#profile';
                }
            }catch(e){
                var root=document.querySelector('#page-content');
                if(root){
                    var box=document.createElement('div');
                    box.className='mb-4 p-4 rounded-lg bg-destructive/10 text-destructive-foreground text-sm';
                    box.textContent='Failed to submit: ' + (e && e.message ? e.message : e);
                    root.prepend(box);
                }
            }finally{
                var root=document.querySelector('#page-content');
                if(root){ root.classList.remove('opacity-70','pointer-events-none'); }
            }
        })(this);"
        class="kt-card-content flex flex-col gap-5 p-0"
      >
        <input type="hidden" name="csrf_token" value="<?= h($csrf) ?>"/>

        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">First Name</label>
            <input name="first_name" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($first_name) ?>" required/>
          </div>
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">Last Name</label>
            <input name="last_name" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($last_name) ?>" required/>
          </div>
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">Phone</label>
            <input name="phone" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($phone) ?>"/>
          </div>
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">Preferred Language</label>
            <select name="preferred_locale" class="w-full rounded-md border border-border px-3 py-2 bg-background">
              <option value="">Keep current (<?= h($preferred_locale ?: 'en') ?>)</option>
              <option value="en" <?= $preferred_locale==='en' ? 'selected' : '' ?>>English</option>
              <option value="de" <?= $preferred_locale==='de' ? 'selected' : '' ?>>German</option>
              <option value="es" <?= $preferred_locale==='es' ? 'selected' : '' ?>>Spanish</option>
            </select>
          </div>
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">Country</label>
            <input name="country" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($country) ?>"/>
          </div>
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">City</label>
            <input name="city" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($city) ?>"/>
          </div>
          <div class="md:col-span-2">
            <label class="kt-form-label font-normal text-mono mb-1 block">Address</label>
            <input name="address_line" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($address_line) ?>"/>
          </div>
          <div>
            <label class="kt-form-label font-normal text-mono mb-1 block">Postal Code</label>
            <input name="postal_code" type="text" class="w-full rounded-md border border-border px-3 py-2 bg-background"
                   value="<?= h($postal_code) ?>"/>
          </div>
        </div>

        <div class="flex justify-center mt-2">
          <button type="submit" class="kt-btn kt-btn-primary">Save Changes</button>
        </div>
      </form>
    </div>
  </div>
</main>

<script>
(function(){
    var INDEX_URL = '/novademo/html/demo1/index.php';

    async function loadChangePassword(ev, linkEl){
        try{
            if (ev) ev.preventDefault();
            var target = linkEl.getAttribute('data-target');
            var root = document.querySelector('#page-content');
            if(root){ root.classList.add('opacity-70','pointer-events-none'); }
            var res = await fetch(target, {credentials:'same-origin', headers:{'X-Requested-With':'fetch'}, cache:'no-store'});
            var txt = await res.text();
            var doc = new DOMParser().parseFromString(txt, 'text/html');
            var fresh = doc.querySelector('#page-content') || doc.body;
            if(fresh && root){ root.replaceWith(fresh); }
            if (history && history.pushState) {
                history.pushState({page:'change-password', params:{}}, '', INDEX_URL + '#change-password');
            } else {
                window.location.href = INDEX_URL + '#change-password';
            }
        }catch(e){
            var root = document.querySelector('#page-content');
            if(root){
                var box=document.createElement('div');
                box.className='mb-4 p-4 rounded-lg bg-destructive/10 text-destructive-foreground text-sm';
                box.textContent='Failed to load Change Password page: ' + (e && e.message ? e.message : e);
                root.prepend(box);
            }
        }finally{
            var root = document.querySelector('#page-content');
            if(root){ root.classList.remove('opacity-70','pointer-events-none'); }
        }
    }

    var headerBtn = document.getElementById('changePasswordLink');
    if(headerBtn){
        headerBtn.addEventListener('click', function(ev){ loadChangePassword(ev, headerBtn); }, {passive:false});
    }
    var inlineBtn = document.getElementById('changePasswordLinkInline');
    if(inlineBtn){
        inlineBtn.addEventListener('click', function(ev){ loadChangePassword(ev, inlineBtn); }, {passive:false});
    }
})();
</script>
</body>
</html>
